// -------------------------------------------------------------------------------
//
// This shader is derived from the Move A.vsh shader as modified by WyldAnimal and
// has been further modfied by TheEmu to be a litle more flexible and also to take
// control parameters as uniform inputs that are themselves defined in  the  scene
// file rather than having to edit the shader to change their values.
//
// -------------------------------------------------------------------------------
//
// Xmovement and Xmovement must be defined with their values in the VGHD .scn file
// using "uniform" clauses in the node invoking this vertex shader. These take the
// form
//    
//    uniform: Xmovement, vec4, type, f, a, p
//    uniform: Ymovement, vec4, type, f, a, p
//    uniform: XYmaximum, vec2, x,y
//
// where type is an integer selecting the type of movement and f, a and p are all
// floating point values parameterising the movement with f being its  frequency,
// a its amplitude and p its phase.
// 
// For their exact meanings see the body of the shader below.

uniform vec2 XYmotion;
uniform vec4 Xparameters;
uniform vec4 Yparameters;

// -------------------------------------------------------------------------------

// Standard shader inputs.
 
uniform float u_Elapsed;    // Elapsed time in seconds
uniform vec2  u_WindowSize; // Window dimensions in pixels

// -------------------------------------------------------------------------------

// Factor_1 provides the main form of the movement.

float Factor_1 ( int type, vec4 params )
 {

   float f = params.x;
   float a = params.y;
   float p = params.z;

   switch ( type )
    { 
      // No displacement

      case 0: return 0.0;

      // Displacement is a function of X co-ordinate

      case 11 : return a * sin ( u_Elapsed*f + p*gl_Vertex.x );
      case 12 : return a * cos ( u_Elapsed*f + p*gl_Vertex.x );
      case 13 : return a * tan ( u_Elapsed*f + p*gl_Vertex.x );

      // Displacement is a function of Y co-ordinate

      case 21 : return a * sin ( u_Elapsed*f + p*gl_Vertex.y );
      case 22 : return a * cos ( u_Elapsed*f + p*gl_Vertex.y );
      case 23 : return a * tan ( u_Elapsed*f + p*gl_Vertex.y );

      // Displacement is a function of X + Y

      case 31 : return a * sin ( u_Elapsed*f + p*(gl_Vertex.x+gl_Vertex.y) );
      case 32 : return a * cos ( u_Elapsed*f + p*(gl_Vertex.x+gl_Vertex.y) );
      case 33 : return a * tan ( u_Elapsed*f + p*(gl_Vertex.x+gl_Vertex.y) );

      // Displacement is a function of X - Y

      case 41 : return a * sin ( u_Elapsed*f + p*(gl_Vertex.x-gl_Vertex.y) );
      case 42 : return a * cos ( u_Elapsed*f + p*(gl_Vertex.x-gl_Vertex.y) );
      case 43 : return a * tan ( u_Elapsed*f + p*(gl_Vertex.x-gl_Vertex.y) );

      // Displacement is a function of length of the position of the vertex

      case 51 : return a * sin ( u_Elapsed*f + p*length(gl_Vertex) );
      case 52 : return a * cos ( u_Elapsed*f + p*length(gl_Vertex) );
      case 53 : return a * tan ( u_Elapsed*f + p*length(gl_Vertex) );

    }

   // Invalid type, but we need to return something

   return 0.0;

 }

// -------------------------------------------------------------------------------

// Factor_2 modulates the movement to provide fixed points.

float Factor_2 ( int type, float f )
 {
   switch ( type )
    { case 0: return 1.0;                  // No fixed points
      case 1: return f;                    // Start point fixed.
      case 2: return 1.0 - f;              // End point fixed.
      case 3: return 1.0 - abs(f-0.5)*2.0; // Start and end fixed.
      case 4: return abs(f-0.5) * 2.0;     // Mid point is fixed.
    }

   // Invalid type, but we need to return something

   return 0.0;

 }

// -------------------------------------------------------------------------------

float Displacement ( int t, vec4 params, float v )
 {
   int t2 = t / 1000;
   int t1 = t - t2*1000;

   return Factor_1(t1,params) * Factor_2(t2,v/params.w);

 }

// -------------------------------------------------------------------------------

void main()
 {

   // Get the displacement for the current vertex.

   vec2 d;

   d.x = Displacement ( int(XYmotion.x), Xparameters, gl_Vertex.x );
   d.y = Displacement ( int(XYmotion.y), Yparameters, gl_Vertex.y );

   // Apply the displacement to the current vertex.

   vec4 Displaced_Vertex = gl_Vertex;
   Displaced_Vertex.xy += d;

   // Update the shader's outputs.

   gl_Position    = gl_ProjectionMatrix * gl_ModelViewMatrix * Displaced_Vertex;
   gl_TexCoord[0] = gl_MultiTexCoord0;
   gl_FrontColor  = gl_Color;

}
